#region Using declarations
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.ComponentModel.DataAnnotations;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Input;
using System.Windows.Media;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Gui;
using NinjaTrader.Gui.Chart;
using NinjaTrader.Gui.SuperDom;
using NinjaTrader.Gui.Tools;
using NinjaTrader.Data;
using NinjaTrader.NinjaScript;
using NinjaTrader.Core.FloatingPoint;
using NinjaTrader.NinjaScript.DrawingTools;
#endregion


public enum FilterMode
        {
            Standard,   // Longs above EMA / Shorts below EMA
            Inverted    // Longs below EMA / Shorts above EMA
        }

namespace NinjaTrader.NinjaScript.Indicators.TradeSaber
{
    public class CandleMACGrok : Indicator
    {
        

        private MACD myMACD;
        private EMA myEMA;

        protected override void OnStateChange()
        {
            if (State == State.SetDefaults)
            {
                Description                                     = @"Candle pattern indicator filtered by MACD diff and optional EMA position.";
                Name                                            = "CandleMACGrok";
                Calculate                                       = Calculate.OnBarClose;
                IsOverlay                                       = false;   // Keep in secondary panel for MACD
                DisplayInDataBox                                = true;
                DrawOnPricePanel                                = false;   // Plots in secondary, but arrows override to price
                DrawHorizontalGridLines                         = true;
                DrawVerticalGridLines                           = true;
                PaintPriceMarkers                               = false;
                ScaleJustification                              = NinjaTrader.Gui.Chart.ScaleJustification.Right;
                IsSuspendedWhileInactive                        = true;

                // NEW: Force transparent plot to show in Data Box
                ShowTransparentPlotsInDataBox                   = true;

                FastLength      = 12;
                SlowLength      = 26;
                SmoothLength    = 9;
                EMAPeriod       = 200;
                EnableEMA       = false;
                EMAFilterMode   = FilterMode.Standard;

                // MACD plots (standard)
                AddPlot(Brushes.DodgerBlue, "MACDLine");
                AddPlot(Brushes.Orange, "MACDAvg");
                AddPlot(new Stroke(Brushes.Lime, 2), PlotStyle.Bar, "Histogram");

                // Entry mode dot plot at 0 line - starts gray, we override color per bar
                AddPlot(new Stroke(Brushes.Gray, 5), PlotStyle.Dot, "EntryMode");

                // Hidden plot for signal value (+1 long, -1 short, 0 none) → will show in Data Box
                AddPlot(Brushes.Transparent, "Signal");
            }
            else if (State == State.DataLoaded)
            {
                myMACD = MACD(Close, FastLength, SlowLength, SmoothLength);
                myEMA  = EMA(Close, EMAPeriod);
            }
        }

        protected override void OnBarUpdate()
        {
            if (CurrentBar < 1) return;

            // MACD values
            Values[0][0] = myMACD[0];
            Values[1][0] = myMACD.Avg[0];
            Values[2][0] = myMACD.Diff[0];

            // EntryMode dot always plotted at 0
            Values[3][0] = 0;

            // Signal starts at 0
            Values[4][0] = 0;

            // Default dot color = gray (filter off / no trade zone)
            PlotBrushes[3][0] = Brushes.Gray;

            // EMA filter → set dot color to indicate allowed direction
            if (EnableEMA)
            {
                bool isAboveEMA = Close[0] > myEMA[0];
                bool isBelowEMA = Close[0] < myEMA[0];

                if (isAboveEMA)
                    PlotBrushes[3][0] = (EMAFilterMode == FilterMode.Standard) ? Brushes.Lime : Brushes.Red;

                else if (isBelowEMA)
                    PlotBrushes[3][0] = (EMAFilterMode == FilterMode.Standard) ? Brushes.Red : Brushes.Lime;

                // equal to EMA → leave gray
            }

            // 2-candle patterns
            bool longPattern = 
                (Close[1] < Open[1])                    // 1st candle red
                && (Close[0] > Open[0])                 // 2nd candle green
                && (Low[0] < Low[1])                    // new low on green candle
                && (High[0] <= High[1])                 // no new high
                && (myMACD.Diff[0] > 0);                // MACD diff positive

            bool shortPattern = 
                (Close[1] > Open[1])                    // 1st candle green
                && (Close[0] < Open[0])                 // 2nd candle red
                && (High[0] > High[1])                  // new high on red candle
                && (Low[0] >= Low[1])                   // no new low
                && (myMACD.Diff[0] < 0);                // MACD diff negative

            // EMA filter allowance
            bool longAllowed = !EnableEMA ||
                (EMAFilterMode == FilterMode.Standard ? Close[0] > myEMA[0] : Close[0] < myEMA[0]);

            bool shortAllowed = !EnableEMA ||
                (EMAFilterMode == FilterMode.Standard ? Close[0] < myEMA[0] : Close[0] > myEMA[0]);

            // Execute signals only if pattern + direction allowed
            if (longPattern && longAllowed)
            {
                // Explicitly draw on price panel
                Draw.ArrowUp(this, "Long" + CurrentBar, true, 0, Low[0] - TickSize * 5, Brushes.Lime, true);
                Values[4][0] = 1;
            }

            if (shortPattern && shortAllowed)
            {
                // Explicitly draw on price panel
                Draw.ArrowDown(this, "Short" + CurrentBar, true, 0, High[0] + TickSize * 5, Brushes.Red, true);
                Values[4][0] = -1;
            }
        }

        #region Properties

        [NinjaScriptProperty]
        [Range(1, int.MaxValue)]
        [Display(Name="Fast Length", Order=1, GroupName="MACD Parameters")]
        public int FastLength { get; set; }

        [NinjaScriptProperty]
        [Range(1, int.MaxValue)]
        [Display(Name="Slow Length", Order=2, GroupName="MACD Parameters")]
        public int SlowLength { get; set; }

        [NinjaScriptProperty]
        [Range(1, int.MaxValue)]
        [Display(Name="Smooth Length", Order=3, GroupName="MACD Parameters")]
        public int SmoothLength { get; set; }

        [NinjaScriptProperty]
        [Range(1, int.MaxValue)]
        [Display(Name="EMA Period", Order=1, GroupName="EMA Filter")]
        public int EMAPeriod { get; set; }

        [NinjaScriptProperty]
        [Display(Name="Enable EMA Filter", Order=2, GroupName="EMA Filter")]
        public bool EnableEMA { get; set; }

        [NinjaScriptProperty]
        [Display(Name="EMA Filter Mode", Order=3, GroupName="EMA Filter")]
        public FilterMode EMAFilterMode { get; set; }

        #endregion
    }
}

#region NinjaScript generated code. Neither change nor remove.

namespace NinjaTrader.NinjaScript.Indicators
{
	public partial class Indicator : NinjaTrader.Gui.NinjaScript.IndicatorRenderBase
	{
		private TradeSaber.CandleMACGrok[] cacheCandleMACGrok;
		public TradeSaber.CandleMACGrok CandleMACGrok(int fastLength, int slowLength, int smoothLength, int eMAPeriod, bool enableEMA, FilterMode eMAFilterMode)
		{
			return CandleMACGrok(Input, fastLength, slowLength, smoothLength, eMAPeriod, enableEMA, eMAFilterMode);
		}

		public TradeSaber.CandleMACGrok CandleMACGrok(ISeries<double> input, int fastLength, int slowLength, int smoothLength, int eMAPeriod, bool enableEMA, FilterMode eMAFilterMode)
		{
			if (cacheCandleMACGrok != null)
				for (int idx = 0; idx < cacheCandleMACGrok.Length; idx++)
					if (cacheCandleMACGrok[idx] != null && cacheCandleMACGrok[idx].FastLength == fastLength && cacheCandleMACGrok[idx].SlowLength == slowLength && cacheCandleMACGrok[idx].SmoothLength == smoothLength && cacheCandleMACGrok[idx].EMAPeriod == eMAPeriod && cacheCandleMACGrok[idx].EnableEMA == enableEMA && cacheCandleMACGrok[idx].EMAFilterMode == eMAFilterMode && cacheCandleMACGrok[idx].EqualsInput(input))
						return cacheCandleMACGrok[idx];
			return CacheIndicator<TradeSaber.CandleMACGrok>(new TradeSaber.CandleMACGrok(){ FastLength = fastLength, SlowLength = slowLength, SmoothLength = smoothLength, EMAPeriod = eMAPeriod, EnableEMA = enableEMA, EMAFilterMode = eMAFilterMode }, input, ref cacheCandleMACGrok);
		}
	}
}

namespace NinjaTrader.NinjaScript.MarketAnalyzerColumns
{
	public partial class MarketAnalyzerColumn : MarketAnalyzerColumnBase
	{
		public Indicators.TradeSaber.CandleMACGrok CandleMACGrok(int fastLength, int slowLength, int smoothLength, int eMAPeriod, bool enableEMA, FilterMode eMAFilterMode)
		{
			return indicator.CandleMACGrok(Input, fastLength, slowLength, smoothLength, eMAPeriod, enableEMA, eMAFilterMode);
		}

		public Indicators.TradeSaber.CandleMACGrok CandleMACGrok(ISeries<double> input , int fastLength, int slowLength, int smoothLength, int eMAPeriod, bool enableEMA, FilterMode eMAFilterMode)
		{
			return indicator.CandleMACGrok(input, fastLength, slowLength, smoothLength, eMAPeriod, enableEMA, eMAFilterMode);
		}
	}
}

namespace NinjaTrader.NinjaScript.Strategies
{
	public partial class Strategy : NinjaTrader.Gui.NinjaScript.StrategyRenderBase
	{
		public Indicators.TradeSaber.CandleMACGrok CandleMACGrok(int fastLength, int slowLength, int smoothLength, int eMAPeriod, bool enableEMA, FilterMode eMAFilterMode)
		{
			return indicator.CandleMACGrok(Input, fastLength, slowLength, smoothLength, eMAPeriod, enableEMA, eMAFilterMode);
		}

		public Indicators.TradeSaber.CandleMACGrok CandleMACGrok(ISeries<double> input , int fastLength, int slowLength, int smoothLength, int eMAPeriod, bool enableEMA, FilterMode eMAFilterMode)
		{
			return indicator.CandleMACGrok(input, fastLength, slowLength, smoothLength, eMAPeriod, enableEMA, eMAFilterMode);
		}
	}
}

#endregion
