#region Using declarations
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.ComponentModel.DataAnnotations;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Input;
using System.Windows.Media;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Gui;
using NinjaTrader.Gui.Chart;
using NinjaTrader.Gui.SuperDom;
using NinjaTrader.Gui.Tools;
using NinjaTrader.Data;
using NinjaTrader.NinjaScript;
using NinjaTrader.Core.FloatingPoint;
using NinjaTrader.NinjaScript.DrawingTools;
#endregion


public enum EMAFilterType
    {
        LongsAboveShortsBelow,
        LongsBelowShortsAbove
    }

namespace NinjaTrader.NinjaScript.Indicators.TradeSaber
{
    public class CandleMACDClaude : Indicator
    {
        #region Variables
        private EMA emaFilter;
        private EMA macdFast;
        private EMA macdSlow;
        
        private Series<double> macdLine;
        private Series<double> macdDiff;
        #endregion

        protected override void OnStateChange()
        {
            if (State == State.SetDefaults)
            {
                Description                             = @"Candle pattern indicator with MACD confirmation and EMA filter";
                Name                                    = "CandleMACDClaude";
                Calculate                               = Calculate.OnBarClose;
                IsOverlay                               = false;
                DisplayInDataBox                        = true;
                DrawOnPricePanel                        = false;
                DrawHorizontalGridLines                 = true;
                DrawVerticalGridLines                   = true;
                PaintPriceMarkers                       = true;
                ScaleJustification                      = NinjaTrader.Gui.Chart.ScaleJustification.Right;
                IsSuspendedWhileInactive                = true;
                
                // MACD Parameters
                MACDFast                                = 12;
                MACDSlow                                = 26;
                MACDSmooth                              = 9;
                
                // EMA Filter Parameters
                UseEMAFilter                            = false;
                EMAPeriod                               = 20;
                EMAFilterMode                           = EMAFilterType.LongsAboveShortsBelow;
                
                // Visual Settings
                ArrowOffset                             = 2;
                
                // Plot Colors
                HistogramPositiveColor                  = Brushes.Green;
                HistogramNegativeColor                  = Brushes.Red;
                LongArrowColor                          = Brushes.Lime;
                ShortArrowColor                         = Brushes.Red;
                NeutralDotColor                         = Brushes.Gray;
                
                AddPlot(new Stroke(Brushes.DodgerBlue, 2), PlotStyle.Line, "MACD");
                AddPlot(new Stroke(Brushes.Orange, 2), PlotStyle.Line, "Signal");
                AddPlot(new Stroke(Brushes.Green, 2), PlotStyle.Bar, "Histogram");
                AddPlot(new Stroke(Brushes.Gray, 2), PlotStyle.Dot, "EMAStatus");
                AddPlot(new Stroke(Brushes.Yellow, 1), PlotStyle.Line, "SignalValue");
                
                AddLine(Brushes.DarkGray, 0, "ZeroLine");
            }
            else if (State == State.Configure)
            {
            }
            else if (State == State.DataLoaded)
            {
                macdLine = new Series<double>(this);
                macdDiff = new Series<double>(this);
                
                macdFast = EMA(Close, MACDFast);
                macdSlow = EMA(Close, MACDSlow);
                emaFilter = EMA(Close, EMAPeriod);
            }
        }

        protected override void OnBarUpdate()
        {
            if (CurrentBar < Math.Max(MACDSlow, EMAPeriod) + 2)
                return;
            
            // Calculate MACD
            double macdValue = macdFast[0] - macdSlow[0];
            macdLine[0] = macdValue;
            
            // Calculate Signal line (EMA of MACD)
            if (CurrentBar < MACDSlow + MACDSmooth)
            {
                Values[1][0] = macdValue;
            }
            else
            {
                Values[1][0] = Values[1][1] + (2.0 / (MACDSmooth + 1)) * (macdValue - Values[1][1]);
            }
            
            double signalValue = Values[1][0];
            double histogram = macdValue - signalValue;
            macdDiff[0] = histogram;
            
            // Set MACD plots
            Values[0][0] = macdValue;      // MACD Line
            Values[2][0] = histogram;       // Histogram
            
            // Color the histogram based on positive/negative
            if (histogram >= 0)
                PlotBrushes[2][0] = HistogramPositiveColor;
            else
                PlotBrushes[2][0] = HistogramNegativeColor;
            
            // Initialize signal value to 0
            Values[4][0] = 0;
            
            // Determine EMA filter status and set dot color
            bool allowLong = true;
            bool allowShort = true;
            
            if (UseEMAFilter)
            {
                double emaValue = emaFilter[0];
                
                if (EMAFilterMode == EMAFilterType.LongsAboveShortsBelow)
                {
                    allowLong = Close[0] > emaValue;
                    allowShort = Close[0] < emaValue;
                }
                else // LongsBelowShortsAbove
                {
                    allowLong = Close[0] < emaValue;
                    allowShort = Close[0] > emaValue;
                }
                
                // Set dot color based on what's allowed
                if (allowLong && !allowShort)
                {
                    PlotBrushes[3][0] = LongArrowColor;
                }
                else if (allowShort && !allowLong)
                {
                    PlotBrushes[3][0] = ShortArrowColor;
                }
                else
                {
                    PlotBrushes[3][0] = NeutralDotColor;
                }
            }
            else
            {
                PlotBrushes[3][0] = NeutralDotColor;
            }
            
            // Set EMA status dot at 0 line
            Values[3][0] = 0;
            
            // Check for candle patterns
            bool longSignal = false;
            bool shortSignal = false;
            
            // Long Condition Check
            // First candle (bar 1) must be red, second candle (bar 0) must be green
            bool firstCandleRed = Close[1] < Open[1];
            bool secondCandleGreen = Close[0] > Open[0];
            bool newLowMade = Low[0] < Low[1];
            bool noNewHigh = High[0] <= High[1];
            bool macdDiffAboveZero = histogram > 0;
            
            if (firstCandleRed && secondCandleGreen && newLowMade && noNewHigh && macdDiffAboveZero)
            {
                if (!UseEMAFilter || allowLong)
                {
                    longSignal = true;
                }
            }
            
            // Short Condition Check
            // First candle (bar 1) must be green, second candle (bar 0) must be red
            bool firstCandleGreen = Close[1] > Open[1];
            bool secondCandleRed = Close[0] < Open[0];
            bool newHighMade = High[0] > High[1];
            bool noNewLow = Low[0] >= Low[1];
            bool macdDiffBelowZero = histogram < 0;
            
            if (firstCandleGreen && secondCandleRed && newHighMade && noNewLow && macdDiffBelowZero)
            {
                if (!UseEMAFilter || allowShort)
                {
                    shortSignal = true;
                }
            }
            
            // Draw arrows and set signal values
            if (longSignal)
            {
                Draw.ArrowUp(this, "LongArrow" + CurrentBar, false, Time[0], Low[0] - (ArrowOffset * TickSize), LongArrowColor, true);
                Values[4][0] = 1;
            }
            
            if (shortSignal)
            {
                Draw.ArrowDown(this, "ShortArrow" + CurrentBar, false, Time[0], High[0] + (ArrowOffset * TickSize), ShortArrowColor, true);
                Values[4][0] = -1;
            }
        }

        #region Properties
        
        [NinjaScriptProperty]
        [Range(1, int.MaxValue)]
        [Display(Name = "MACD Fast", Description = "Fast EMA period for MACD", Order = 1, GroupName = "MACD Parameters")]
        public int MACDFast { get; set; }

        [NinjaScriptProperty]
        [Range(1, int.MaxValue)]
        [Display(Name = "MACD Slow", Description = "Slow EMA period for MACD", Order = 2, GroupName = "MACD Parameters")]
        public int MACDSlow { get; set; }

        [NinjaScriptProperty]
        [Range(1, int.MaxValue)]
        [Display(Name = "MACD Smooth", Description = "Signal line smoothing period", Order = 3, GroupName = "MACD Parameters")]
        public int MACDSmooth { get; set; }

        [NinjaScriptProperty]
        [Display(Name = "Use EMA Filter", Description = "Enable/disable EMA filter", Order = 1, GroupName = "EMA Filter")]
        public bool UseEMAFilter { get; set; }

        [NinjaScriptProperty]
        [Range(1, int.MaxValue)]
        [Display(Name = "EMA Period", Description = "Period for EMA filter", Order = 2, GroupName = "EMA Filter")]
        public int EMAPeriod { get; set; }

        [NinjaScriptProperty]
        [Display(Name = "EMA Filter Mode", Description = "How to filter signals based on EMA", Order = 3, GroupName = "EMA Filter")]
        public EMAFilterType EMAFilterMode { get; set; }

        [NinjaScriptProperty]
        [Range(1, int.MaxValue)]
        [Display(Name = "Arrow Offset", Description = "Arrow offset in ticks from high/low", Order = 1, GroupName = "Visual Settings")]
        public int ArrowOffset { get; set; }

        [XmlIgnore]
        [Display(Name = "MACD Line Color", Description = "Color of MACD line", Order = 1, GroupName = "Plot Colors")]
        public Brush MACDLineColor
        {
            get { return Plots[0].Brush; }
            set { Plots[0].Brush = value; }
        }

        [Browsable(false)]
        public string MACDLineColorSerializable
        {
            get { return Serialize.BrushToString(MACDLineColor); }
            set { MACDLineColor = Serialize.StringToBrush(value); }
        }

        [XmlIgnore]
        [Display(Name = "Signal Line Color", Description = "Color of Signal line", Order = 2, GroupName = "Plot Colors")]
        public Brush SignalLineColor
        {
            get { return Plots[1].Brush; }
            set { Plots[1].Brush = value; }
        }

        [Browsable(false)]
        public string SignalLineColorSerializable
        {
            get { return Serialize.BrushToString(SignalLineColor); }
            set { SignalLineColor = Serialize.StringToBrush(value); }
        }

        [XmlIgnore]
        [Display(Name = "Histogram Positive Color", Description = "Color of positive histogram bars", Order = 3, GroupName = "Plot Colors")]
        public Brush HistogramPositiveColor { get; set; }

        [Browsable(false)]
        public string HistogramPositiveColorSerializable
        {
            get { return Serialize.BrushToString(HistogramPositiveColor); }
            set { HistogramPositiveColor = Serialize.StringToBrush(value); }
        }

        [XmlIgnore]
        [Display(Name = "Histogram Negative Color", Description = "Color of negative histogram bars", Order = 4, GroupName = "Plot Colors")]
        public Brush HistogramNegativeColor { get; set; }

        [Browsable(false)]
        public string HistogramNegativeColorSerializable
        {
            get { return Serialize.BrushToString(HistogramNegativeColor); }
            set { HistogramNegativeColor = Serialize.StringToBrush(value); }
        }

        [XmlIgnore]
        [Display(Name = "Long Arrow Color", Description = "Color of long signal arrows", Order = 5, GroupName = "Plot Colors")]
        public Brush LongArrowColor { get; set; }

        [Browsable(false)]
        public string LongArrowColorSerializable
        {
            get { return Serialize.BrushToString(LongArrowColor); }
            set { LongArrowColor = Serialize.StringToBrush(value); }
        }

        [XmlIgnore]
        [Display(Name = "Short Arrow Color", Description = "Color of short signal arrows", Order = 6, GroupName = "Plot Colors")]
        public Brush ShortArrowColor { get; set; }

        [Browsable(false)]
        public string ShortArrowColorSerializable
        {
            get { return Serialize.BrushToString(ShortArrowColor); }
            set { ShortArrowColor = Serialize.StringToBrush(value); }
        }

        [XmlIgnore]
        [Display(Name = "Neutral Dot Color", Description = "Color of neutral/disabled EMA status dot", Order = 7, GroupName = "Plot Colors")]
        public Brush NeutralDotColor { get; set; }

        [Browsable(false)]
        public string NeutralDotColorSerializable
        {
            get { return Serialize.BrushToString(NeutralDotColor); }
            set { NeutralDotColor = Serialize.StringToBrush(value); }
        }

        // Plot accessors
        [Browsable(false)]
        [XmlIgnore]
        public Series<double> MACDPlot
        {
            get { return Values[0]; }
        }

        [Browsable(false)]
        [XmlIgnore]
        public Series<double> SignalPlot
        {
            get { return Values[1]; }
        }

        [Browsable(false)]
        [XmlIgnore]
        public Series<double> HistogramPlot
        {
            get { return Values[2]; }
        }

        [Browsable(false)]
        [XmlIgnore]
        public Series<double> EMAStatusPlot
        {
            get { return Values[3]; }
        }

        [Browsable(false)]
        [XmlIgnore]
        public Series<double> SignalValuePlot
        {
            get { return Values[4]; }
        }

        #endregion
    }
}

#region NinjaScript generated code. Neither change nor remove.

namespace NinjaTrader.NinjaScript.Indicators
{
	public partial class Indicator : NinjaTrader.Gui.NinjaScript.IndicatorRenderBase
	{
		private TradeSaber.CandleMACDClaude[] cacheCandleMACDClaude;
		public TradeSaber.CandleMACDClaude CandleMACDClaude(int mACDFast, int mACDSlow, int mACDSmooth, bool useEMAFilter, int eMAPeriod, EMAFilterType eMAFilterMode, int arrowOffset)
		{
			return CandleMACDClaude(Input, mACDFast, mACDSlow, mACDSmooth, useEMAFilter, eMAPeriod, eMAFilterMode, arrowOffset);
		}

		public TradeSaber.CandleMACDClaude CandleMACDClaude(ISeries<double> input, int mACDFast, int mACDSlow, int mACDSmooth, bool useEMAFilter, int eMAPeriod, EMAFilterType eMAFilterMode, int arrowOffset)
		{
			if (cacheCandleMACDClaude != null)
				for (int idx = 0; idx < cacheCandleMACDClaude.Length; idx++)
					if (cacheCandleMACDClaude[idx] != null && cacheCandleMACDClaude[idx].MACDFast == mACDFast && cacheCandleMACDClaude[idx].MACDSlow == mACDSlow && cacheCandleMACDClaude[idx].MACDSmooth == mACDSmooth && cacheCandleMACDClaude[idx].UseEMAFilter == useEMAFilter && cacheCandleMACDClaude[idx].EMAPeriod == eMAPeriod && cacheCandleMACDClaude[idx].EMAFilterMode == eMAFilterMode && cacheCandleMACDClaude[idx].ArrowOffset == arrowOffset && cacheCandleMACDClaude[idx].EqualsInput(input))
						return cacheCandleMACDClaude[idx];
			return CacheIndicator<TradeSaber.CandleMACDClaude>(new TradeSaber.CandleMACDClaude(){ MACDFast = mACDFast, MACDSlow = mACDSlow, MACDSmooth = mACDSmooth, UseEMAFilter = useEMAFilter, EMAPeriod = eMAPeriod, EMAFilterMode = eMAFilterMode, ArrowOffset = arrowOffset }, input, ref cacheCandleMACDClaude);
		}
	}
}

namespace NinjaTrader.NinjaScript.MarketAnalyzerColumns
{
	public partial class MarketAnalyzerColumn : MarketAnalyzerColumnBase
	{
		public Indicators.TradeSaber.CandleMACDClaude CandleMACDClaude(int mACDFast, int mACDSlow, int mACDSmooth, bool useEMAFilter, int eMAPeriod, EMAFilterType eMAFilterMode, int arrowOffset)
		{
			return indicator.CandleMACDClaude(Input, mACDFast, mACDSlow, mACDSmooth, useEMAFilter, eMAPeriod, eMAFilterMode, arrowOffset);
		}

		public Indicators.TradeSaber.CandleMACDClaude CandleMACDClaude(ISeries<double> input , int mACDFast, int mACDSlow, int mACDSmooth, bool useEMAFilter, int eMAPeriod, EMAFilterType eMAFilterMode, int arrowOffset)
		{
			return indicator.CandleMACDClaude(input, mACDFast, mACDSlow, mACDSmooth, useEMAFilter, eMAPeriod, eMAFilterMode, arrowOffset);
		}
	}
}

namespace NinjaTrader.NinjaScript.Strategies
{
	public partial class Strategy : NinjaTrader.Gui.NinjaScript.StrategyRenderBase
	{
		public Indicators.TradeSaber.CandleMACDClaude CandleMACDClaude(int mACDFast, int mACDSlow, int mACDSmooth, bool useEMAFilter, int eMAPeriod, EMAFilterType eMAFilterMode, int arrowOffset)
		{
			return indicator.CandleMACDClaude(Input, mACDFast, mACDSlow, mACDSmooth, useEMAFilter, eMAPeriod, eMAFilterMode, arrowOffset);
		}

		public Indicators.TradeSaber.CandleMACDClaude CandleMACDClaude(ISeries<double> input , int mACDFast, int mACDSlow, int mACDSmooth, bool useEMAFilter, int eMAPeriod, EMAFilterType eMAFilterMode, int arrowOffset)
		{
			return indicator.CandleMACDClaude(input, mACDFast, mACDSlow, mACDSmooth, useEMAFilter, eMAPeriod, eMAFilterMode, arrowOffset);
		}
	}
}

#endregion
